/* $Id: ttltrg2.c,v 1.4 1998/09/16 02:05:25 ericb Exp $ */
/* Copyright (C) 1997, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test ability to drive once-per-trigger TTLTRG line */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	64
#define	BLOCKSIZE	1024

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: ttltrg2.c,v 1.4 1998/09/16 02:05:25 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int manual, int data_mode)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, data_mode));

    /* Force e1432_init_measure to use our clock and trigger line
       setups.  Otherwise it would override them with its own
       settings. */
    CHECK(e1432_set_auto_group_meas(hw, group,
				    E1432_AUTO_GROUP_MEAS_OFF));

    /* Pick TTLTRG lines for trigger and gclock which don't conflict
       with defaults for satrg and clock.  The "trigger" setting, in
       this case TTLTRG line 2, is the one which will toggle once per
       measurement loop on the E143x modules. */
    CHECK(e1432_set_ttltrg_trigger(hw, group, E1432_TTLTRG_2));
    CHECK(e1432_set_ttltrg_gclock(hw, group, E1432_TTLTRG_3));

    /* Set up group clock and trigger lines for all but the one which
       will actually drive the clock and trigger lines. */
    CHECK(e1432_set_clock_master(hw, group, E1432_MASTER_CLOCK_OFF));
    CHECK(e1432_set_clock_source(hw, group, E1432_CLOCK_SOURCE_VXI));
    CHECK(e1432_set_multi_sync(hw, group, E1432_MULTI_SYNC_ON));

    /* Now pick last channel to be the one to drive clock and trigger
       lines */
    CHECK(e1432_set_clock_master(hw, chan_list[nchan - 1],
				 E1432_MASTER_CLOCK_ON));
    CHECK(e1432_set_clock_source(hw, chan_list[nchan - 1],
				 E1432_CLOCK_SOURCE_INTERNAL));
    CHECK(e1432_set_multi_sync(hw, chan_list[nchan - 1],
			       E1432_MULTI_SYNC_VXD));

    if (manual)
	CHECK(e1432_set_auto_trigger(hw, group, E1432_MANUAL_TRIGGER));

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Block available found\n");

    return 0;
}

static int
get_scan(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 int verbose, double span, int manual)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     chan;

    /* Wait for block available */
    if (manual)
	/* Allow for really long wait */
	CHECK(wait_block_avail(hw, group, verbose, BLOCKSIZE, 0.000001));
    else
	CHECK(wait_block_avail(hw, group, verbose, BLOCKSIZE, span));

    /* Read the data */
    for (chan = 0; chan < nchan; chan++)
    {
	if (verbose > 3)
	    (void) printf("Reading chan %d\n", chan);

	CHECK(e1432_read_float64_data(hw, chan_list[chan],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, &trailer,
				      &count));
	if (count != BLOCKSIZE)
	{
	    (void) fprintf(stderr, "%s: e1432_read_float64_data: "
			   "actual count was %ld, expected %d\n",
			   progname, count, BLOCKSIZE);
	    return -1;
	}
    }

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int manual, int verbose)
{
    int     i;

    CHECK(e1432_init_measure(hw, group));
    for (i = 0; i < 10; i++)
	CHECK(get_scan(hw, group, nchan, chan_list,
		       verbose, 20000, manual));
    return 0;
}

static int
cleanup(E1432ID hw, int group)
{
    CHECK(e1432_finish_measure(hw, group));
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-bcfmuvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-b: Use block mode\n"
		   "\t-c: Use continuous mode\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Use manual trigger\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, data_mode, manual, nmod, verbose;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    laddr[0] = 8;
    manual = 0;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "bcL:mn:N:uvV")) != -1)
	switch (c)
	{
	case 'b':
	    data_mode = E1432_BLOCK_MODE;
	    break;
	case 'c':
	    data_mode = E1432_CONTINUOUS_MODE;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    manual = 1;
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, manual, data_mode) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, manual, verbose) < 0)
	return EXIT_FAILURE;
    if (cleanup(hw, group) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
